# Security Fixes Implementation Guide

## Overview

This document outlines the security fixes implemented to address SQL injection and XSS vulnerabilities while preserving Bricks Builder and other page builder functionality.

## Critical Issues Fixed

### 1. SQL Injection Vulnerabilities

**Files Fixed:**
- `templates/admin-main.php` (Line 10)
- `uninstall.php` (Lines 21, 47-48)
- `includes/Restore/DatabaseImporter.php` (Security validation improvements)

**Solutions:**
- Used `esc_sql()` for safe table name construction
- Implemented `$wpdb->prepare()` for parameterized queries
- Enhanced security validation without blocking legitimate restore operations

### 2. XSS Vulnerability

**File Fixed:**
- `templates/admin-schedule.php` (Line 380)

**Solution:**
- Added `esc_html()` around `print_r()` output to prevent XSS

### 3. Serialized Data Corruption (Bricks Builder Issue)

**Root Cause:**
- Security fixes were corrupting PHP serialized data by applying SQL escaping to serialized strings
- URL replacement wasn't updating byte counts in serialized strings (e.g., `s:60:"url"` where 60 is byte length)

**New Components Added:**
- `includes/Restore/SerializedDataHandler.php` - Safe serialized data migration
- Enhanced `includes/Compatibility/BricksHandler.php` integration
- Updated `includes/Restore/UrlMigrator.php` to use SerializedDataHandler

## Key Improvements

### SerializedDataHandler Features

1. **Safe Deserialization**: Attempts clean unserialize first
2. **Repair Capabilities**: Fixes corrupted serialized data with wrong byte counts
3. **Fallback Methods**: Multiple repair strategies for edge cases
4. **Validation**: Ensures migrated data remains valid serialized format

### Enhanced DatabaseImporter Security

1. **Expanded Whitelist**: More restore operations allowed (REPLACE INTO, CREATE INDEX, etc.)
2. **Better Table Validation**: More permissive for restore while blocking dangerous names
3. **Detailed Logging**: Error logging for debugging rejected statements

### Bricks Builder Protection

1. **Serialized Data Integrity**: URLs migrated without corruption
2. **Cache Management**: Proper cache invalidation after migration
3. **Multiple Repair Strategies**: Aggressive repair for corrupted data

## Testing Your Installation

### Method 1: Automated Testing

1. Access your WordPress admin
2. Go to: `your-site.com/wp-admin/?run_jbm_tests=1`
3. Review the test results

### Method 2: Manual Testing

1. **Create a test backup** with Bricks Builder content
2. **Restore to a different domain** (local development site)
3. **Verify Bricks layouts** are intact and functional
4. **Check page builder content** displays correctly

### Method 3: Code Review

```php
// Test serialized data handling
$handler = new \JenvaBackupMigration\Restore\SerializedDataHandler();
$test_data = serialize(['url' => 'https://oldsite.com/image.jpg']);
$migrated = $handler->migrateUrls($test_data, 'https://oldsite.com', 'https://newsite.com');
$result = unserialize($migrated); // Should succeed
```

## Migration Best Practices

### Before Migration

1. **Clear all caches** on source site
2. **Verify Bricks data integrity** (Admin → Bricks → Settings → Cleanup)
3. **Run database optimization**

### During Migration

1. **Use "Full" backup type** to ensure all data is captured
2. **Enable debug logging** if issues occur
3. **Monitor memory usage** for large sites

### After Migration

1. **Clear all caches** (WordPress + Bricks + any cache plugins)
2. **Regenerate permalinks** (Settings → Permalinks → Save)
3. **Re-save Bricks settings** (Bricks → Settings → Save)
4. **Test page builder editing** functionality

## Troubleshooting

### If Bricks Layouts Are Still Broken

1. **Run the repair tool:**
   ```php
   $handler = new \JenvaBackupMigration\Compatibility\BricksHandler();
   $result = $handler->repairAll();
   ```

2. **Manually clear Bricks cache:**
   - Delete `/wp-content/uploads/bricks/` folder
   - Go to Bricks → Settings → Cleanup → Clear all CSS files

3. **Check for remaining old URLs:**
   ```sql
   SELECT * FROM wp_postmeta WHERE meta_value LIKE '%oldsite.com%' AND meta_key LIKE '_bricks_%';
   SELECT * FROM wp_options WHERE option_value LIKE '%oldsite.com%' AND option_name LIKE 'bricks_%';
   ```

### If Security Warnings Persist

1. **Check DatabaseImporter logs:**
   - Look for "JBM: Rejected SQL statement" in error logs
   - Verify legitimate statements aren't being blocked

2. **Review template output:**
   - Ensure all `echo` statements use proper escaping
   - Check for unescaped variables in debug output

## Performance Considerations

### For Large Sites

1. **Increase memory limits:**
   ```php
   ini_set('memory_limit', '2048M');
   ini_set('max_execution_time', '0');
   ```

2. **Use CLI for migrations** when possible
3. **Process in smaller chunks** if timeouts occur

### Database Optimization

1. **Index key columns** used in URL migration
2. **Optimize tables** before and after migration
3. **Monitor query performance** during URL replacement

## Security Validation

### SQL Injection Protection

The enhanced validation allows these legitimate operations:
- `CREATE TABLE` / `CREATE INDEX`
- `INSERT INTO` / `REPLACE INTO`  
- `UPDATE` / `DELETE FROM`
- `ALTER TABLE` / `DROP TABLE`
- Transaction control (`START TRANSACTION`, `COMMIT`, etc.)

While blocking dangerous operations:
- `DROP DATABASE` / `CREATE DATABASE`
- `LOAD_FILE()` / `INTO OUTFILE`
- `GRANT` / `REVOKE`
- File system access functions

### XSS Prevention

All debug output is now properly escaped:
- `print_r()` output wrapped in `esc_html()`
- Variable output uses appropriate escaping functions
- No raw data echoed to browser

## Compatibility

### Supported Page Builders

1. **Bricks Builder** ✅ Full support with specialized handler
2. **Elementor** ✅ Supported via BuilderHandler
3. **Beaver Builder** ✅ Supported via BuilderHandler  
4. **Divi Builder** ✅ Supported via BuilderHandler
5. **Visual Composer** ✅ Supported via BuilderHandler
6. **Oxygen Builder** ✅ Supported via BuilderHandler

### WordPress Compatibility

- **WordPress 5.8+** ✅ Fully supported
- **PHP 7.4+** ✅ Required
- **MySQL 5.6+** ✅ Recommended

## Changelog

### Version 1.0.1 (Security Fix)

- ✅ Fixed SQL injection in templates and uninstall script
- ✅ Fixed XSS vulnerability in debug output  
- ✅ Added SerializedDataHandler for safe URL migration
- ✅ Enhanced DatabaseImporter security validation
- ✅ Improved Bricks Builder compatibility
- ✅ Added comprehensive testing framework

This implementation ensures your backup and migration plugin remains secure while preserving full functionality with Bricks Builder and other page builders.